/*  _ _ _
 * | (_) |__ ___ ___ _ __  __ _   SEPA library  www.libsepa.com
 * | | | '_ (_-</ -_) '_ \/ _` |  Copyright (c) 2013-2014 Keppler IT GmbH.
 * |_|_|_.__/__/\___| .__/\__,_|____________________________________________
 *                  |_|
 * $Id: sepa.h 177 2014-01-14 17:19:18Z kk $
 */

/*! \file sepa.h
 *  \brief libsepa header (C/C++ API)
 *
 *  This include file defines all exported functions required for using libsepa.
 */

#ifndef __SEPA_H
#define __SEPA_H

#include <stdint.h>

#ifdef __cplusplus
extern "C" {
#endif

#define SEPA_BIC_LENGTH 11
#define SEPA_IBAN_MAXLENGTH 34		/*!< maximum length of a valid IBAN number */

typedef struct sepa_t sepa_t;		/*!< abstract type for a libsepa instance handle */

typedef enum {
	SEPA_ERROR = 0,
	SEPA_OK = 1
} sepa_status_t;

typedef enum {
	SEPA_LOOKUP_OK			= 1,
	SEPA_LOOKUP_NOTSUPPORTED = 0,
	SEPA_LOOKUP_FORMATERROR = -1,
	SEPA_LOOKUP_NOTFOUND	= -2,
	SEPA_LOOKUP_FAILED		= -3,
	SEPA_LOOKUP_AMBIGUOUS	= -4
} sepa_lookup_status_t;

typedef enum {
	SEPA_INIT_LICUSER		= 1,	/*!< set license user */
	SEPA_INIT_LICCODE		= 2,	/*!< set license code */
	SEPA_INIT_RULEBOOK		= 3,	/*!< select SEPA rulebook */
} sepa_init_t;

typedef enum {
	SEPA_MSGTYPE_UNKNOWN	= 0,
	SEPA_MSGTYPE_DDI		= 1,	/*!< Direct Debit Initiation - pain.008.003.02 */
	SEPA_MSGTYPE_CTI		= 2		/*!< Credit Transfer Initiation - pain.001.003.03 */
} sepa_msgtype_t;

typedef enum {
	SEPA_DDTYPE_CORE	= 0,		/*!< Direct Debit 'LocalInstrumentCode': CORE */
	SEPA_DDTYPE_COR1	= 1,		/*!< Direct Debit 'LocalInstrumentCode': COR1 */
	SEPA_DDTYPE_B2B		= 2			/*!< Direct Debit 'LocalInstrumentCode': B2B */
} sepa_ddtype_t;

typedef struct {
	const char *key;
	const char *value;
} sepa_keyvalue_t;

typedef struct {
	char *name;
	char *zip;
	char *city;
	char *country;
	char *bic;
	uint16_t flags;
	char __priv[];
} sepa_bankinfo_t;

#define SEPA_SCL_SCT	1		/*!< sepa_bankinfo_t.flags: support for SEPA cash transfer (according to SCL) */
#define SEPA_SCL_SDD	2		/*!< sepa_bankinfo_t.flags: support for SEPA direct debit (according to SCL) */
#define SEPA_SCL_COR1	4		/*!< sepa_bankinfo_t.flags: support for SEPA direct debit COR1 (according to SCL) */
#define SEPA_SCL_B2B	8		/*!< sepa_bankinfo_t.flags: support for SEPA direct debit B2B (according to SCL) */

#define SEPA_VERSION "1.8"
#define SEPA_VERSION_MAJOR 1
#define SEPA_VERSION_MINOR 8
#define SEPA_VERSION_INT 10800
extern const char *sepa_version_str;
extern const int sepa_version_int;

sepa_status_t sepa_init(sepa_init_t option, const char *value);
sepa_t *sepa_new(sepa_msgtype_t msgtype);
void sepa_free(sepa_t *s);
char *sepa_toXML(sepa_t *s);
sepa_status_t sepa_setIBAN(sepa_t *s, const char *iban);
sepa_status_t sepa_setBIC(sepa_t *s, const char *bic);
sepa_status_t sepa_setName(sepa_t *s, const char *name);
sepa_status_t sepa_setCreditorIdentifier(sepa_t *s, const char *ci);
sepa_status_t sepa_setDate(sepa_t *s, const char *date);
sepa_status_t sepa_setDDType(sepa_t *s, sepa_ddtype_t ddtype);
const char *sepa_getError(sepa_t *s);
sepa_status_t sepa_add(sepa_t *s, sepa_keyvalue_t kv[]);
sepa_lookup_status_t sepa_iban_getBank(const char *iban, sepa_bankinfo_t **info);
sepa_lookup_status_t sepa_iban_convert(const char *country, const char *account, const char *bankid, char iban[SEPA_IBAN_MAXLENGTH+1]);
sepa_lookup_status_t sepa_iban_check(const char *iban);
sepa_lookup_status_t sepa_iban_getBIC(const char *iban, char bic[SEPA_BIC_LENGTH+1]);
sepa_lookup_status_t sepa_bic_getBank(const char *bic, sepa_bankinfo_t **info);			/*!< return data from SCL for this BIC (bank name & SCL flags) */

/* these functions are used rather internally: */
sepa_status_t sepa_account_check(const char *account, unsigned char rule);
sepa_status_t sepa_account_check2(const char *account, const char *bankid, unsigned char rule);
sepa_lookup_status_t sepa_bankid_check(const char *country, const char *bankid, sepa_bankinfo_t **info);
void sepa_cleanup(void);

#ifdef __cplusplus
} // extern "C" {...}
#endif

#endif /* !__SEPA_H */

/* <EOF>_________________________________________________________________ */
